//	TorusGames2DTicTacToe.c
//
//	The board's nine cells are indexed as
//
//		(0,2)	(1,2)	(2,2)
//		(0,1)	(1,1)	(2,1)
//		(0,0)	(1,0)	(2,0)
//
//	to agree with Metal's bottom-up Normalized Device Coordinates,
//	at the expense of violating the usual (row, column)
//	convention for arrays.  Cell (0,0) straddles all four corners.
//
//	© 2021 by Jeff Weeks
//	See TermsOfUse.txt

#include "TorusGames-Common.h"
#include "GeometryGamesUtilities-Common.h"
#include "GeometryGamesSound.h"
#include <math.h>


#define NUM_2D_BACKGROUND_TEXTURE_REPETITIONS_TIC_TAC_TOE	3

//	The 'X' and 'O' markers get inset within the cell.
#define MARKER_SIZE_FACTOR	0.75

#define TIC_TAC_TOE_WIN_LINE_THICKNESS	0.0625

#define ONE_THIRD	0.33333333333333333333


//	Public functions with private names
static void				TicTacToeReset(ModelData *md);
static bool				TicTacToeDragBegin(ModelData *md, bool aRightClick);
static void				TicTacToeDragObject(ModelData *md, double aHandLocalDeltaH, double aHandLocalDeltaV);
static void				TicTacToeDragEnd(ModelData *md, double aDragDuration, bool aTouchSequenceWasCancelled);
static void				TicTacToeSimulationUpdate(ModelData *md);

//	Private functions
static void				FindHitCell(Placement2D *aHandPlacement, TopologyType aTopology, unsigned int *h, unsigned int *v, bool *aFlip);
static void				ComputerMoves(ModelData *md);
static bool				ComputerMovesToWin(ModelData *md, TicTacToePlayer aPlayer);
static bool				ComputerMovesToBlock(ModelData *md,TicTacToePlayer aPlayer);
static bool				ComputerMovesRandomly(ModelData *md, TicTacToePlayer aPlayer);
static TicTacToePlayer	CheckForWin(ModelData *md, Placement2D *aThreeInARow);
static void				NormalizeCell(signed int *h, signed int *v, TopologyType aTopology);


void TicTacToe2DSetUp(ModelData *md)
{
	unsigned int	h,
					v;

	//	Initialize function pointers.
	md->itsGameShutDown					= NULL;
	md->itsGameReset					= &TicTacToeReset;
	md->itsGameHumanVsComputerChanged	= NULL;
	md->itsGame2DHandMoved				= NULL;
	md->itsGame2DDragBegin				= &TicTacToeDragBegin;
	md->itsGame2DDragObject				= &TicTacToeDragObject;
	md->itsGame2DDragEnd				= &TicTacToeDragEnd;
	md->itsGame3DDragBegin				= NULL;
	md->itsGame3DDragObject				= NULL;
	md->itsGame3DDragEnd				= NULL;
	md->itsGame3DGridSize				= NULL;
	md->itsGameCharacterInput			= NULL;
	md->itsGameSimulationUpdate			= &TicTacToeSimulationUpdate;
	md->itsGameRefreshMessage			= NULL;

	//	Initialize variables.

	for (h = 0; h < 3; h++)
		for (v = 0; v < 3; v++)
			md->itsGameOf.TicTacToe2D.itsCellFlip[h][v] = false;
	
	md->itsGameOf.TicTacToe2D.itsHitCellH = 0;
	md->itsGameOf.TicTacToe2D.itsHitCellV = 0;

	md->itsGameOf.TicTacToe2D.itsWinLine.itsH		= 0.0;
	md->itsGameOf.TicTacToe2D.itsWinLine.itsV		= 0.0;
	md->itsGameOf.TicTacToe2D.itsWinLine.itsFlip	= false;
	md->itsGameOf.TicTacToe2D.itsWinLine.itsAngle	= 0.0;
	md->itsGameOf.TicTacToe2D.itsWinLine.itsSizeH	= 0.0;
	md->itsGameOf.TicTacToe2D.itsWinLine.itsSizeV	= 0.0;
	
	//	Set up the board.
	TicTacToeReset(md);
}


static void TicTacToeReset(ModelData *md)
{
	unsigned int	h,
					v;

	for (h = 0; h < 3; h++)
		for (v = 0; v < 3; v++)
			md->itsGameOf.TicTacToe2D.itsBoard[h][v] = PlayerNone;

#ifdef MAKE_GAME_CHOICE_ICONS
	md->itsGameOf.TicTacToe2D.itsBoard[1][1] = PlayerX;
	md->itsGameOf.TicTacToe2D.itsBoard[2][2] = PlayerO;
#endif
#ifdef GAME_CONTENT_FOR_SCREENSHOT
	md->itsGameOf.TicTacToe2D.itsBoard[0][1] = PlayerX;
	md->itsGameOf.TicTacToe2D.itsBoard[0][2] = PlayerO;
	md->itsGameOf.TicTacToe2D.itsBoard[1][0] = PlayerO;
	md->itsGameOf.TicTacToe2D.itsBoard[1][2] = PlayerX;
	md->itsGameOf.TicTacToe2D.itsBoard[2][1] = PlayerO;
	md->itsGameOf.TicTacToe2D.itsBoard[2][2] = PlayerX;
#endif

	//	Abort any pending simulation.
	SimulationEnd(md);

	md->itsGameIsOver						= false;
	md->itsFlashFlag						= false;
	md->itsGameOf.TicTacToe2D.itsWhoseTurn	= PlayerX;

#ifdef GAME_CONTENT_FOR_SCREENSHOT
	CheckForWin(md, &md->itsGameOf.TicTacToe2D.itsWinLine);
	md->itsGameIsOver						= true;
#endif
}


static bool TicTacToeDragBegin(
	ModelData	*md,
	bool		aRightClick)
{
	double			theIntegerPart;	//	value is ignored
	unsigned int	h,
					v;
	bool			theFlip;

	UNUSED_PARAMETER(aRightClick);

	//	If the game is already over, ignore all hits.
	if (md->itsGameIsOver)
		return false;

	//	Are we near the center of a cell?
	//	If not, treat this hit as a scroll.
	if (modf(3.0*(md->its2DHandPlacement.itsH + 0.5) + 0.25, &theIntegerPart) > 0.5
	 || modf(3.0*(md->its2DHandPlacement.itsV + 0.5) + 0.25, &theIntegerPart) > 0.5)
	{
		return false;
	}

	//	Which cell got hit?
	FindHitCell(&md->its2DHandPlacement, md->itsTopology, &h, &v, &theFlip);

	//	If the cell is already occupied, treat this hit as a scroll.
	if (md->itsGameOf.TicTacToe2D.itsBoard[h][v] != PlayerNone)
		return false;

	//	During a content drag, remember which cell was hit originally,
	//	and accept the Tic-Tac-Toe move iff the drag ends over that same cell.
	//	This gives the user a chance to change his/her mind,
	//	and also helps protect against accidentally making a move
	//	when trying to scroll.
	md->itsGameOf.TicTacToe2D.itsHitCellH = h;
	md->itsGameOf.TicTacToe2D.itsHitCellV = v;

	return true;
}


static void TicTacToeDragObject(
	ModelData	*md,
	double		aHandLocalDeltaH,
	double		aHandLocalDeltaV)
{
	md->its2DHandPlacement.itsH += (md->its2DHandPlacement.itsFlip ? -aHandLocalDeltaH : aHandLocalDeltaH);
	md->its2DHandPlacement.itsV += aHandLocalDeltaV;
	Normalize2DPlacement(&md->its2DHandPlacement, md->itsTopology);
}


static void TicTacToeDragEnd(
	ModelData	*md,
	double		aDragDuration,	//	in seconds
	bool		aTouchSequenceWasCancelled)
{
	unsigned int	h,
					v;
	bool			theFlip;
	
	UNUSED_PARAMETER(aDragDuration);

	//	If a touch sequence got cancelled (perhaps because a gesture was recognized),
	//	return without making a move.
	if (aTouchSequenceWasCancelled)
		return;

	//	In which cell did the drag end?
	FindHitCell(&md->its2DHandPlacement, md->itsTopology, &h, &v, &theFlip);

	//	Accept the move iff the drag ended in the same cell where it began.
	if (h == md->itsGameOf.TicTacToe2D.itsHitCellH
	 && v == md->itsGameOf.TicTacToe2D.itsHitCellV)
	{
		//	Mark O or X in the hit cell, according to whose turn it is.
		md->itsGameOf.TicTacToe2D.itsBoard[h][v] = md->itsGameOf.TicTacToe2D.itsWhoseTurn;

		//	The orientation of the symbol relative to the fundamental domain 
		//	should match the orientation of the nearest image of the hand
		//	relative to the fundamental domain.
		md->itsGameOf.TicTacToe2D.itsCellFlip[h][v] = theFlip;

		//	Toggle itsGameOf.TicTacToe2D.itsWhoseTurn.
		md->itsGameOf.TicTacToe2D.itsWhoseTurn = (md->itsGameOf.TicTacToe2D.itsWhoseTurn == PlayerX) ? PlayerO : PlayerX;

		//	Play the move sound.
		EnqueueSoundRequest(u"TicTacToeMove.mid");

		//	Has somebody won?
		if (CheckForWin(md, &md->itsGameOf.TicTacToe2D.itsWinLine) != PlayerNone)
		{
			SimulationBegin(md, Simulation2DTicTacToeWaitToDisplayWin);
		}
		else
		{
			if (md->itsHumanVsComputer)
			{
				//	Wait half a second before letting the computer respond,
				//	for the user's psychological benefit.
				SimulationBegin(md, Simulation2DTicTacToeWaitToMove);
			}
		}
	}
}

static void FindHitCell(
	Placement2D		*aHandPlacement,	//	input
	TopologyType	aTopology,			//	input
	unsigned int	*h,					//	output
	unsigned int	*v,					//	output
	bool			*aFlip)				//	output
{
	//	Find the integer coordinates (h,v) of the hit cell.
	//
	//	While we're at it, note the chirality of the hand.
	//	When recording chirality, cells with coordinates h = 0 or/and v = 0
	//	are considered to lie along the left and/or bottom edges
	//	of the fundamental domain (rather than the right and/or top edges).

	*h		= (unsigned int) floor(3.0*(aHandPlacement->itsH + 2.0/3.0));
	*v		= (unsigned int) floor(3.0*(aHandPlacement->itsV + 2.0/3.0));
	*aFlip	= aHandPlacement->itsFlip;

	if (*v > 2)	//	Test for *v > 2 rather than *v == 3 to protect against invalid input.
	{
		*v = 0;
		if (aTopology == Topology2DKlein)
		{
			*h = 3 - *h;
			*aFlip = ! *aFlip;
		}
	}
	if (*h > 2)	//	Test for *h > 2 rather than *h == 3 to protect against invalid input.
	{
		*h = 0;
	}
}


static void ComputerMoves(ModelData *md)
{
	//	Start the move sound playing.
	EnqueueSoundRequest(u"TicTacToeMove.mid");

	if (ComputerMovesToWin(md, md->itsGameOf.TicTacToe2D.itsWhoseTurn))
	{
		//	Record the win and flash the winning line.
		SimulationBegin(md, Simulation2DTicTacToeWaitToDisplayWin);
	}
	else
	{
		if ( ! ComputerMovesToBlock(md, md->itsGameOf.TicTacToe2D.itsWhoseTurn) )
			(void) ComputerMovesRandomly(md, md->itsGameOf.TicTacToe2D.itsWhoseTurn);
	}

	md->itsGameOf.TicTacToe2D.itsWhoseTurn = (md->itsGameOf.TicTacToe2D.itsWhoseTurn == PlayerX) ? PlayerO : PlayerX;
}


static bool ComputerMovesToWin(
	ModelData		*md,
	TicTacToePlayer	aPlayer)
{
	unsigned int	h,
					v;

	for (h = 0; h < 3; h++)
		for (v = 0; v < 3; v++)
			if (md->itsGameOf.TicTacToe2D.itsBoard[h][v] == PlayerNone)
			{
				md->itsGameOf.TicTacToe2D.itsBoard[h][v] = aPlayer;
				if (CheckForWin(md, &md->itsGameOf.TicTacToe2D.itsWinLine) == aPlayer)
					return true;
				else
					md->itsGameOf.TicTacToe2D.itsBoard[h][v] = PlayerNone;
			}

	return false;
}


static bool ComputerMovesToBlock(
	ModelData		*md,
	TicTacToePlayer	aPlayer)
{
	TicTacToePlayer	theOpponent;
	unsigned int	h,
					v;

	theOpponent = (aPlayer == PlayerX ? PlayerO : PlayerX);

	for (h = 0; h < 3; h++)
		for (v = 0; v < 3; v++)
			if (md->itsGameOf.TicTacToe2D.itsBoard[h][v] == PlayerNone)
			{
				md->itsGameOf.TicTacToe2D.itsBoard[h][v] = theOpponent;
				if (CheckForWin(md, NULL) == theOpponent)
				{
					md->itsGameOf.TicTacToe2D.itsBoard[h][v] = aPlayer;
					return true;
				}
				else
					md->itsGameOf.TicTacToe2D.itsBoard[h][v] = PlayerNone;
			}

	return false;
}


static bool ComputerMovesRandomly(
	ModelData		*md,
	TicTacToePlayer	aPlayer)
{
	unsigned int	theNumEmptyCells,
					h,
					v,
					theCount;

	//	Count the number of empty cells.
	theNumEmptyCells = 0;
	for (h = 0; h < 3; h++)
		for (v = 0; v < 3; v++)
			if (md->itsGameOf.TicTacToe2D.itsBoard[h][v] == PlayerNone)
				theNumEmptyCells++;

	if (theNumEmptyCells == 0)
		return false;	//	should never occur

	//	Move randomly.
	theCount = RandomUnsignedInteger() % theNumEmptyCells;

	for (h = 0; h < 3; h++)
		for (v = 0; v < 3; v++)
			if (md->itsGameOf.TicTacToe2D.itsBoard[h][v] == PlayerNone)
				if (theCount-- == 0)
				{
					md->itsGameOf.TicTacToe2D.itsBoard[h][v] = aPlayer;
					return true;
				}

	return false;	//	should never occur
}


static TicTacToePlayer CheckForWin(
	ModelData	*md,
	Placement2D	*aThreeInARow)	//	may be NULL
{
	signed int		h,	//	Signed integers allow for ±1 offsets.
					v,
					h1,
					v1,
					h2,
					v2;
	unsigned int	d;

	//	The winning three-in-a-row may go in any of four
	//	possible directions: E-W, SE-NW, S-N, SW-NE.
	static const signed int	theWinDirections[4][2]	= { {-1, 0}, {-1, 1}, { 0, 1}, { 1, 1} };
	static const double		theWinAngles[4]			= { 0.0,  -0.25*PI, 0.5*PI, 0.25*PI },
							theWinLengths[4]		= { 1.0,    ROOT2,    1.0,   ROOT2  };

	//	Consider each possible center (h,v) for the three-in-a-row.
	for (h = 0; h < 3; h++)
	{
		for (v = 0; v < 3; v++)
		{
			//	Ignore empty cells.
			if (md->itsGameOf.TicTacToe2D.itsBoard[h][v] == PlayerNone)
				continue;

			//	Consider each possible direction.
			for (d = 0; d < 4; d++)
			{
				//	Let (h1,v1) be the square in front of us
				//	in the given direction, and (h2,v2) be
				//	the square behind us in that direction.
				//	If both agree with (h,v), we've got a win.

				h1 = h + theWinDirections[d][0];
				v1 = v + theWinDirections[d][1];
				NormalizeCell(&h1, &v1, md->itsTopology);
				if (md->itsGameOf.TicTacToe2D.itsBoard[h1][v1] != md->itsGameOf.TicTacToe2D.itsBoard[h][v])
					continue;

				h2 = h - theWinDirections[d][0];
				v2 = v - theWinDirections[d][1];
				NormalizeCell(&h2, &v2, md->itsTopology);
				if (md->itsGameOf.TicTacToe2D.itsBoard[h2][v2] != md->itsGameOf.TicTacToe2D.itsBoard[h][v])
					continue;

				//	We've got a win!
				if (aThreeInARow != NULL)
				{
					aThreeInARow->itsH		= -0.5 + ONE_THIRD*h;
					aThreeInARow->itsV		= -0.5 + ONE_THIRD*v;
					aThreeInARow->itsFlip	= false;	//	ignored
					aThreeInARow->itsAngle	= theWinAngles[d];
					aThreeInARow->itsSizeH	= theWinLengths[d];					//	line length
					aThreeInARow->itsSizeV	= TIC_TAC_TOE_WIN_LINE_THICKNESS;	//	line thickness
				}
				return md->itsGameOf.TicTacToe2D.itsBoard[h][v];
			}
		}
	}

	//	Nobody's won yet.
	return PlayerNone;
}


static void NormalizeCell(
	signed int		*h,
	signed int		*v,
	TopologyType	aTopology)
{
	if (*v == 3)
	{
		*v = 0;
		if (aTopology == Topology2DKlein)
			*h = 3 - *h;
	}
	if (*v == -1)
	{
		*v = 2;
		if (aTopology == Topology2DKlein)
			*h = 3 - *h;
	}

	if (*h > 2)
		*h -= 3;
	if (*h < 0)
		*h += 3;
}


static void TicTacToeSimulationUpdate(ModelData *md)
{
	switch (md->itsSimulationStatus)
	{
		case Simulation2DTicTacToeWaitToMove:
			//	Wait half a second before letting the computer move.
			if (md->itsSimulationElapsedTime >= 0.5)
			{
				SimulationEnd(md);
				ComputerMoves(md);
			}
			break;

		case Simulation2DTicTacToeWaitToDisplayWin:
			//	Pause briefly so the user may appreciate
			//	the move before the winning line appears.
			if (md->itsSimulationElapsedTime >= 0.5)
			{
				SimulationEnd(md);
				md->itsGameIsOver = true;							//	Record the win.
				EnqueueSoundRequest(u"TicTacToeWin.mid");					//	Start a victory sound playing.
				SimulationBegin(md, Simulation2DTicTacToeFlash);	//	Flash the three-in-a-row marker.
			}
			break;

		case Simulation2DTicTacToeFlash:
			md->itsFlashFlag = (((unsigned int) floor(10.0 * md->itsSimulationElapsedTime)) % 2) ? true : false;
			if (md->itsSimulationElapsedTime >= 1.4)
			{
				SimulationEnd(md);
				md->itsFlashFlag = false;
			}
			break;

		default:
			break;
	}
}


unsigned int GetNum2DTicTacToeBackgroundTextureRepetitions(void)
{
	return NUM_2D_BACKGROUND_TEXTURE_REPETITIONS_TIC_TAC_TOE;
}

void Get2DTicTacToeKleinAxisColors(
	float	someKleinAxisColors[2][4])	//	output;  premultiplied alpha
{
	someKleinAxisColors[0][0] = 0.75;
	someKleinAxisColors[0][1] = 1.00;
	someKleinAxisColors[0][2] = 0.50;
	someKleinAxisColors[0][3] = 1.00;

	someKleinAxisColors[1][0] = 0.25;
	someKleinAxisColors[1][1] = 0.00;
	someKleinAxisColors[1][2] = 0.50;
	someKleinAxisColors[1][3] = 1.00;
}

unsigned int GetNum2DTicTacToeSprites(void)
{
	return 3*3 + 3*3 + 1;	//	full cells, inset cells and (possibily unused) win line
}

void Get2DTicTacToeSpritePlacements(
	ModelData		*md,				//	input
	unsigned int	aNumSprites,		//	input
	Placement2D		*aPlacementBuffer)	//	output;  big enough to hold aNumSprites Placement2D's
{
	unsigned int	i,
					ii,
					h,
					v;

	GEOMETRY_GAMES_ASSERT(
		md->itsGame == Game2DTicTacToe,
		"Game2DTicTacToe must be active");

	GEOMETRY_GAMES_ASSERT(
		aNumSprites == GetNum2DTicTacToeSprites(),
		"Internal error:  wrong buffer size");

	//	cells
	//		note (h,v) indexing -- not (row,col)
	for (h = 0; h < 3; h++)
	{
		for (v = 0; v < 3; v++)
		{
			//	full cell
			i = 3*h + v;
			aPlacementBuffer[i].itsH		= -0.5 + ONE_THIRD * (double)h;
			aPlacementBuffer[i].itsV		= -0.5 + ONE_THIRD * (double)v;
			aPlacementBuffer[i].itsFlip		= md->itsGameOf.TicTacToe2D.itsCellFlip[h][v];
			aPlacementBuffer[i].itsAngle	= 0.0;
			aPlacementBuffer[i].itsSizeH	= ONE_THIRD;
			aPlacementBuffer[i].itsSizeV	= ONE_THIRD;
			
			//	marker within cell
			ii = 3*3 + i;
			aPlacementBuffer[ii] = aPlacementBuffer[i];
			aPlacementBuffer[ii].itsSizeH *= MARKER_SIZE_FACTOR;
			aPlacementBuffer[ii].itsSizeV *= MARKER_SIZE_FACTOR;
		}
	}

	//	win line
	if (md->itsGameIsOver)
		aPlacementBuffer[3*3 + 3*3 + 0] = md->itsGameOf.TicTacToe2D.itsWinLine;
	else
		aPlacementBuffer[3*3 + 3*3 + 0] = gUnusedPlacement;
}
